require 'spec_helper'

provider_class = Puppet::Type.type(:user).provider(:aix)

describe provider_class do

  let(:lsuser_all_example) do
    <<-OUTPUT
root id=0 pgrp=system groups=system,bin,sys,security,cron,audit,lp home=/root shell=/usr/bin/bash auditclasses=general login=true su=true rlogin=true daemon=true admin=true sugroups=ALL admgroups=lolt,allstaff tpath=nosak ttys=ALL expires=0 auth1=SYSTEM auth2=NONE umask=22 registry=files SYSTEM=compat logintimes= loginretries=0 pwdwarntime=0 account_locked=false minage=0 maxage=0 maxexpired=-1 minalpha=0 minother=0 mindiff=0 maxrepeats=8 minlen=0 histexpire=0 histsize=0 pwdchecks= dictionlist= default_roles= fsize=2097151 cpu=-1 data=262144 stack=65536 core=2097151 rss=65536 nofiles=2000 time_last_login=1358465855 time_last_unsuccessful_login=1358378454 tty_last_login=ssh tty_last_unsuccessful_login=ssh host_last_login=rpm-builder.puppetlabs.lan host_last_unsuccessful_login=192.168.100.78 unsuccessful_login_count=0 roles=
guest id=100 pgrp=usr groups=usr home=/home/guest login=true su=true rlogin=true daemon=true admin=false sugroups=ALL admgroups= tpath=nosak ttys=ALL expires=0 auth1=SYSTEM auth2=NONE umask=22 registry=files SYSTEM=compat logintimes= loginretries=0 pwdwarntime=0 account_locked=false minage=0 maxage=0 maxexpired=-1 minalpha=0 minother=0 mindiff=0 maxrepeats=8 minlen=0 histexpire=0 histsize=0 pwdchecks= dictionlist= default_roles= fsize=2097151 cpu=-1 data=262144 stack=65536 core=2097151 rss=65536 nofiles=2000 roles=
    OUTPUT
  end

  let(:lsgroup_all_example) do
    <<-OUTPUT
root id=0 pgrp=system groups=system,bin,sys,security,cron,audit,lp home=/root shell=/usr/bin/bash
guest id=100 pgrp=usr groups=usr home=/home/guest
    OUTPUT
  end

  before do
    @resource = stub('resource')
    @provider = provider_class.new(@resource)
  end

  it "should be able to return a group name based on a group ID" do
    @provider.stubs(:lsgroupscmd)

    @provider.stubs(:execute).returns(lsgroup_all_example)

    @provider.groupname_by_id(100).should == 'guest'
  end

  it "should be able to list all users" do
    provider_class.stubs(:command)

    provider_class.stubs(:execute).returns(lsuser_all_example)

    provider_class.list_all.should == ['root', 'guest']
  end

  describe "#managed_attribute_keys" do
    let(:existing_attributes) do
      { :account_locked => 'false',
        :admin => 'false',
        :login => 'true',
        'su' => 'true'
      }
    end

    before(:each) do
      original_parameters = { :attributes => attribute_array }
      @resource.stubs(:original_parameters).returns(original_parameters)
    end

    describe "invoked via manifest" do
      let(:attribute_array) { ["rlogin=false", "login =true"] }

      it "should return only the keys of the attribute key=value pair from manifest" do
        keys = @provider.managed_attribute_keys(existing_attributes)
        keys.should be_include(:rlogin)
        keys.should be_include(:login)
        keys.should_not be_include(:su)
      end

      it "should strip spaces from symbols" do
        keys = @provider.managed_attribute_keys(existing_attributes)
        keys.should be_include(:login)
        keys.should_not be_include(:"login ")
      end

      it "should have the same count as that from the manifest" do
        keys = @provider.managed_attribute_keys(existing_attributes)
        keys.size.should == attribute_array.size
      end

      it "should convert the keys to symbols" do
        keys = @provider.managed_attribute_keys(existing_attributes)
        all_symbols = keys.all? {|k| k.is_a? Symbol}
        all_symbols.should be_true
      end
    end

    describe "invoked via RAL" do
      let(:attribute_array) { nil }

      it "should return the keys in supplied hash" do
        keys = @provider.managed_attribute_keys(existing_attributes)
        keys.should_not be_include(:rlogin)
        keys.should be_include(:login)
        keys.should be_include(:su)
      end

      it "should convert the keys to symbols" do
        keys = @provider.managed_attribute_keys(existing_attributes)
        all_symbols = keys.all? {|k| k.is_a? Symbol}
        all_symbols.should be_true
      end
    end
  end

  describe "#should_include?" do
    it "should exclude keys translated into something else" do
      managed_keys = [:rlogin]
      @provider.class.attribute_mapping_from.stubs(:include?).with(:rlogin).returns(true)
      @provider.class.stubs(:attribute_ignore).returns([])
      @provider.should_include?(:rlogin, managed_keys).should be_false
    end

    it "should exclude keys explicitly ignored" do
      managed_keys = [:rlogin]
      @provider.class.attribute_mapping_from.stubs(:include?).with(:rlogin).returns(false)
      @provider.class.stubs(:attribute_ignore).returns([:rlogin])
      @provider.should_include?(:rlogin, managed_keys).should be_false
    end

    it "should exclude keys not specified in manifest" do
      managed_keys = [:su]
      @provider.class.attribute_mapping_from.stubs(:include?).with(:rlogin).returns(false)
      @provider.class.stubs(:attribute_ignore).returns([])
      @provider.should_include?(:rlogin, managed_keys).should be_false
    end

    it "should include keys specified in manifest if not translated or ignored" do
      managed_keys = [:rlogin]
      @provider.class.attribute_mapping_from.stubs(:include?).with(:rlogin).returns(false)
      @provider.class.stubs(:attribute_ignore).returns([])
      @provider.should_include?(:rlogin, managed_keys).should be_true
    end
  end
end
