(ns puppetlabs.puppetdb.cli.import
  "Import utility

   This is a command-line tool for importing data into PuppetDB. It expects
   as input a tarball generated by the PuppetDB `export` command-line tool."
  (:require [me.raynes.fs :as fs]
            [puppetlabs.puppetdb.cli.export :as cli-export]
            [clj-http.client :as http-client]
            [puppetlabs.puppetdb.cheshire :as json]
            [puppetlabs.puppetdb.utils :as utils]
            [puppetlabs.puppetdb.archive :as archive]
            [puppetlabs.kitchensink.core :as kitchensink]
            [puppetlabs.puppetdb.export :as export]
            [clojure.java.io :as io]
            [slingshot.slingshot :refer [try+ throw+]]))

(def cli-description "Import PuppetDB catalog data from a backup file")

(def metadata-path
  (.getPath (io/file utils/export-root-dir export/export-metadata-file-name)))

(defn parse-metadata
  "Parses the export metadata file to determine, e.g., what versions of the
  commands should be used during import."
  [tarball]
  {:post [(map? %)
          (:command_versions %)]}
  (with-open [tar-reader (archive/tarball-reader tarball)]
    (when-not (archive/find-entry tar-reader metadata-path)
      (throw (IllegalStateException.
              (format "Unable to find export metadata file '%s' in archive '%s'"
                      metadata-path
                      tarball))))
    (utils/read-json-content tar-reader true)))

(defn- validate-cli!
  [args]
  (let [specs [["-i" "--infile INFILE" "Path to backup file (required)"]
               ["-H" "--host HOST" "Hostname of PuppetDB server"
                :default "127.0.0.1"]
               ["-p" "--port PORT" "Port to connect to PuppetDB server (HTTP protocol only)"
                :default 8080
                :parse-fn #(Integer/parseInt %)]]
        required [:infile]
        validate-file-exists! (fn [{:keys [infile] :as options}]
                             (when-not (fs/exists? infile)
                               (throw+ {:type ::cli-help
                                        :message (format "Import from %s failed. File not found." infile)}))
                             options)
        construct-base-url (fn [{:keys [host port] :as options}]
                             (-> options
                                 (assoc :base-url (utils/pdb-admin-base-url host port cli-export/admin-api-version))
                                 (dissoc :host :port)))]
    (utils/try+-process-cli!
     (fn []
       (-> args
           (kitchensink/cli! specs required)
           first
           validate-file-exists!
           construct-base-url
           utils/validate-cli-base-url!)))))

(defn -main
  [& args]
  (let [{:keys [infile base-url]} (validate-cli! args)
        import-archive (fs/normalized-path infile)
        command-versions (:command_versions (parse-metadata import-archive))]
    (try
      (println " Importing " infile " to PuppetDB...")
      (http-client/post (str (utils/base-url->str base-url) "/archive")
                        {:multipart [{:name "Content/type" :content "application/octet-stream"}
                                     {:name "archive" :content import-archive}
                                     {:name "command_versions" :content (json/generate-pretty-string command-versions)}]})
      (println " Finished importing " infile " to PuppetDB.")
      (catch Throwable e
        (println e "Error importing " infile)))))
